<?php

declare(strict_types=1);

namespace App\Controllers;

use DateTime;

class DividendsController
{
    public static function index(): void
    {
        $portfolioId = (int)request_input('portfolio_id');
        if (!$portfolioId) {
            flash('Missing portfolio_id.', 'danger');
            redirect('/portfolio');
        }

        $dividends = fetch_all(
            'SELECT d.*, s.symbol FROM dividend d JOIN stock s ON s.id = d.stock_id WHERE d.portfolio_id = :pid ORDER BY d.payment_date DESC',
            ['pid' => $portfolioId]
        );

        if (request_input('export') !== null) {
            self::exportDividends($dividends);
        }

        render('dividends/index', [
            'dividends' => $dividends,
            'portfolio_id' => $portfolioId,
        ]);
    }

    public static function import(): void
    {
        $portfolioId = (int)request_input('portfolio_id');
        if (!$portfolioId) {
            flash('Missing portfolio_id.', 'danger');
            redirect('/portfolio');
        }

        $file = $_FILES['file'] ?? null;
        if (!$file || $file['error'] !== UPLOAD_ERR_OK) {
            flash('Select a CSV file first.', 'danger');
            redirect('/dividends?portfolio_id=' . $portfolioId);
        }

        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if ($ext !== 'csv') {
            flash('Only .csv files are supported.', 'warning');
            redirect('/dividends?portfolio_id=' . $portfolioId);
        }

        try {
            $rows = self::readCsv($file['tmp_name']);
            $required = [
                'Security Symbol', 'Security Name', 'Financial Year',
                'Net Dividend Amount (PKR)', 'Payment Date',
            ];

            if (empty($rows) || array_slice(array_keys($rows[0]), 0, 5) !== $required) {
                throw new \RuntimeException('Template columns/order must be: ' . implode(', ', $required));
            }

            db()->beginTransaction();
            execute('DELETE FROM dividend WHERE portfolio_id = :pid', ['pid' => $portfolioId]);

            foreach ($rows as $row) {
                $stock = self::getOrCreateStock((string)$row['Security Symbol'], (string)($row['Security Name'] ?? ''));

                $netAmount = (float)str_replace(',', '', (string)$row['Net Dividend Amount (PKR)']);
                $payDate = self::parseDate($row['Payment Date'] ?? null, 'd/m/Y');

                execute(
                    'INSERT INTO dividend (portfolio_id, stock_id, financial_year, net_amount, payment_date, created_at, updated_at)
                     VALUES (:pid, :sid, :fy, :net, :date, NOW(), NOW())',
                    [
                        'pid' => $portfolioId,
                        'sid' => $stock['id'],
                        'fy' => (string)$row['Financial Year'],
                        'net' => $netAmount,
                        'date' => $payDate,
                    ]
                );
            }

            db()->commit();
            flash('Dividends imported successfully (missing stocks were auto-added).', 'success');
        } catch (\Throwable $e) {
            db()->rollBack();
            flash('Import failed: ' . $e->getMessage(), 'danger');
        }

        redirect('/dividends?portfolio_id=' . $portfolioId);
    }

    public static function create(): void
    {
        $portfolioId = (int)request_input('portfolio_id');
        if (!$portfolioId) {
            flash('Missing portfolio_id.', 'danger');
            redirect('/portfolio');
        }

        if (request_method() === 'POST') {
            try {
                $symbol = strtoupper(trim((string)request_input('symbol', '')));
                $stock = fetch_one('SELECT id FROM stock WHERE symbol = :symbol', ['symbol' => $symbol]);
                if (!$stock) {
                    throw new \RuntimeException('Stock not found.');
                }

                $paymentDate = self::parseDate(request_input('payment_date'), 'd/m/Y');

                execute(
                    'INSERT INTO dividend (portfolio_id, stock_id, financial_year, net_amount, payment_date, created_at, updated_at)
                     VALUES (:pid, :sid, :fy, :net, :date, NOW(), NOW())',
                    [
                        'pid' => $portfolioId,
                        'sid' => $stock['id'],
                        'fy' => request_input('financial_year'),
                        'net' => request_input('net_amount'),
                        'date' => $paymentDate,
                    ]
                );

                flash('Dividend added.', 'success');
                redirect('/dividends?portfolio_id=' . $portfolioId);
            } catch (\Throwable $e) {
                flash($e->getMessage(), 'danger');
            }
        }

        render('dividends/new', ['portfolio_id' => $portfolioId]);
    }

    public static function edit(string $id): void
    {
        $div = fetch_one('SELECT * FROM dividend WHERE id = :id', ['id' => $id]);
        if (!$div) {
            redirect('/dividends');
        }

        if (request_method() === 'POST') {
            try {
                $paymentDate = self::parseDate(request_input('payment_date'), 'd/m/Y');
                execute(
                    'UPDATE dividend SET financial_year = :fy, net_amount = :net, payment_date = :date, updated_at = NOW() WHERE id = :id',
                    [
                        'fy' => request_input('financial_year'),
                        'net' => request_input('net_amount'),
                        'date' => $paymentDate,
                        'id' => $id,
                    ]
                );
                flash('Dividend updated.', 'success');
                redirect('/dividends?portfolio_id=' . $div['portfolio_id']);
            } catch (\Throwable $e) {
                flash($e->getMessage(), 'danger');
            }
        }

        render('dividends/edit', ['div' => $div]);
    }

    public static function delete(string $id): void
    {
        $div = fetch_one('SELECT portfolio_id FROM dividend WHERE id = :id', ['id' => $id]);
        if ($div) {
            execute('DELETE FROM dividend WHERE id = :id', ['id' => $id]);
            flash('Dividend deleted.', 'success');
            redirect('/dividends?portfolio_id=' . $div['portfolio_id']);
        }
        redirect('/dividends');
    }

    private static function getOrCreateStock(string $symbol, string $name): array
    {
        $symbol = strtoupper(trim($symbol));
        $stock = fetch_one('SELECT id FROM stock WHERE symbol = :symbol', ['symbol' => $symbol]);
        if ($stock) {
            return $stock;
        }

        execute('INSERT INTO stock (symbol, sector) VALUES (:symbol, NULL)', ['symbol' => $symbol]);
        return ['id' => (int)db()->lastInsertId()];
    }

    private static function parseDate(mixed $value, string $format): string
    {
        if ($value instanceof DateTime) {
            return $value->format('Y-m-d');
        }
        $date = DateTime::createFromFormat($format, (string)$value);
        if ($date) {
            return $date->format('Y-m-d');
        }
        $date = date_create((string)$value);
        return $date ? $date->format('Y-m-d') : date('Y-m-d');
    }

    private static function readCsv(string $path): array
    {
        $handle = fopen($path, 'r');
        if ($handle === false) {
            throw new \RuntimeException('Unable to open CSV file.');
        }

        $headers = fgetcsv($handle);
        if ($headers === false) {
            fclose($handle);
            return [];
        }

        $headers = array_map(static fn($h) => trim((string)$h), $headers);
        $rows = [];
        while (($row = fgetcsv($handle)) !== false) {
            if (count($row) === 1 && trim((string)$row[0]) === '') {
                continue;
            }
            $entry = [];
            foreach ($headers as $i => $header) {
                $entry[$header] = $row[$i] ?? null;
            }
            $rows[] = $entry;
        }
        fclose($handle);
        return $rows;
    }

    private static function exportDividends(array $dividends): void
    {
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="dividends.csv"');
        $out = fopen('php://output', 'w');
        fputcsv($out, ['Symbol', 'Financial Year', 'Net Amount', 'Payment Date']);
        foreach ($dividends as $div) {
            fputcsv($out, [
                $div['symbol'] ?? '',
                $div['financial_year'] ?? '',
                $div['net_amount'] ?? '',
                $div['payment_date'] ?? '',
            ]);
        }
        fclose($out);
        exit;
    }
}
