<?php

declare(strict_types=1);

namespace App\Controllers;

class JsCallsController
{
    public static function uploadDaily(): void
    {
        if (request_method() === 'POST') {
            $file = $_FILES['file'] ?? null;
            $today = date('Y-m-d');

            if ($file && $file['error'] === UPLOAD_ERR_OK && !str_ends_with(strtolower($file['name']), '.csv')) {
                flash('Invalid file format. Upload a .csv file.', 'danger');
                redirect('/js-calls/upload-daily');
            }

            if ($file && $file['error'] === UPLOAD_ERR_OK && str_ends_with(strtolower($file['name']), '.csv')) {
                try {
                    $rows = self::readCsv($file['tmp_name']);
                    $inserted = 0;
                    $skipped = 0;

                    foreach ($rows as $row) {
                        $symbol = strtoupper(trim((string)($row['Symbol'] ?? '')));
                        if ($symbol === '') {
                            continue;
                        }
                        $exists = fetch_one(
                            'SELECT id FROM js_daily_calls WHERE symbol = :symbol AND DATE(uploaded_at) = :date',
                            ['symbol' => $symbol, 'date' => $today]
                        );
                        if ($exists) {
                            $skipped++;
                            continue;
                        }

                        execute(
                            'INSERT INTO js_daily_calls (symbol, strategy, close_price, s1, pivot, r1, rsi_14, uploaded_at)
                             VALUES (:symbol, :strategy, :close, :s1, :pivot, :r1, :rsi, :uploaded)',
                            [
                                'symbol' => $symbol,
                                'strategy' => $row['Strategy'] ?? null,
                                'close' => $row['Close'] ?? null,
                                's1' => $row['S1'] ?? null,
                                'pivot' => $row['PIVOT'] ?? null,
                                'r1' => $row['R1'] ?? null,
                                'rsi' => $row['14-DRSI'] ?? null,
                                'uploaded' => $today,
                            ]
                        );
                        $inserted++;
                    }

                    flash("{$inserted} daily calls uploaded, {$skipped} skipped (duplicates).", 'success');
                } catch (\Throwable $e) {
                    flash('Upload failed: ' . $e->getMessage(), 'danger');
                }

                redirect('/js-calls/upload-daily');
            }

            $symbol = strtoupper(trim((string)request_input('symbol', '')));
            if ($symbol === '') {
                flash('Symbol is required.', 'warning');
                redirect('/js-calls/upload-daily');
            }

            $exists = fetch_one(
                'SELECT id FROM js_daily_calls WHERE symbol = :symbol AND DATE(uploaded_at) = :date',
                ['symbol' => $symbol, 'date' => $today]
            );
            if ($exists) {
                flash("Call for {$symbol} already exists today.", 'warning');
                redirect('/js-calls/upload-daily');
            }

            try {
                execute(
                    'INSERT INTO js_daily_calls (symbol, strategy, close_price, s1, pivot, r1, rsi_14, uploaded_at)
                     VALUES (:symbol, :strategy, :close, :s1, :pivot, :r1, :rsi, :uploaded)',
                    [
                        'symbol' => $symbol,
                        'strategy' => request_input('strategy'),
                        'close' => request_input('close_price'),
                        's1' => request_input('s1'),
                        'pivot' => request_input('pivot'),
                        'r1' => request_input('r1'),
                        'rsi' => request_input('rsi_14'),
                        'uploaded' => $today,
                    ]
                );
                flash("Call added for {$symbol}.", 'success');
            } catch (\Throwable $e) {
                flash('Manual entry failed: ' . $e->getMessage(), 'danger');
            }

            redirect('/js-calls/upload-daily');
        }

        render('calls/upload_daily_calls');
    }

    public static function listDaily(): void
    {
        $calls = fetch_all('SELECT * FROM js_daily_calls ORDER BY uploaded_at DESC, symbol');
        render('calls/list_daily_calls', ['calls' => $calls]);
    }

    public static function editDaily(string $id): void
    {
        $call = fetch_one('SELECT * FROM js_daily_calls WHERE id = :id', ['id' => $id]);
        if (!$call) {
            redirect('/js-calls/daily-calls');
        }

        if (request_method() === 'POST') {
            try {
                execute(
                    'UPDATE js_daily_calls SET symbol = :symbol, strategy = :strategy, close_price = :close, s1 = :s1, pivot = :pivot, r1 = :r1, rsi_14 = :rsi WHERE id = :id',
                    [
                        'symbol' => strtoupper(trim((string)request_input('symbol', ''))),
                        'strategy' => request_input('strategy'),
                        'close' => request_input('close_price'),
                        's1' => request_input('s1'),
                        'pivot' => request_input('pivot'),
                        'r1' => request_input('r1'),
                        'rsi' => request_input('rsi_14'),
                        'id' => $id,
                    ]
                );
                flash('Call updated successfully.', 'success');
                redirect('/js-calls/daily-calls');
            } catch (\Throwable $e) {
                flash('Update failed: ' . $e->getMessage(), 'danger');
            }
        }

        render('calls/edit_daily_call', ['call' => $call]);
    }

    public static function deleteDaily(string $id): void
    {
        execute('DELETE FROM js_daily_calls WHERE id = :id', ['id' => $id]);
        flash('Call deleted.', 'success');
        redirect('/js-calls/daily-calls');
    }

    public static function uploadPivot(): void
    {
        if (request_method() === 'POST') {
            $file = $_FILES['file'] ?? null;
            $today = date('Y-m-d');

            if ($file && $file['error'] === UPLOAD_ERR_OK && !str_ends_with(strtolower($file['name']), '.csv')) {
                flash('Invalid file format. Upload a .csv file.', 'danger');
                redirect('/js-calls/upload-pivot');
            }

            if ($file && $file['error'] === UPLOAD_ERR_OK && str_ends_with(strtolower($file['name']), '.csv')) {
                try {
                    $rows = self::readCsv($file['tmp_name']);
                    $inserted = 0;
                    $skipped = 0;

                    foreach ($rows as $row) {
                        $symbol = strtoupper(trim((string)($row['Symbol'] ?? '')));
                        if ($symbol === '') {
                            continue;
                        }
                        $exists = fetch_one(
                            'SELECT id FROM js_pivot_calls WHERE symbol = :symbol AND DATE(uploaded_at) = :date',
                            ['symbol' => $symbol, 'date' => $today]
                        );
                        if ($exists) {
                            $skipped++;
                            continue;
                        }

                        execute(
                            'INSERT INTO js_pivot_calls (symbol, close_price, volume, rsi_14, rsi_signal, macd_signal, s3, s2, s1, pivot, r1, r2, r3, uploaded_at)
                             VALUES (:symbol, :close, :volume, :rsi, :rsi_signal, :macd, :s3, :s2, :s1, :pivot, :r1, :r2, :r3, :uploaded)',
                            [
                                'symbol' => $symbol,
                                'close' => $row['Close'] ?? null,
                                'volume' => $row['Volume'] ?? null,
                                'rsi' => $row['RSI 14'] ?? null,
                                'rsi_signal' => $row['RSI Signal'] ?? null,
                                'macd' => $row['MACD Signal'] ?? null,
                                's3' => $row['S3'] ?? null,
                                's2' => $row['S2'] ?? null,
                                's1' => $row['S1'] ?? null,
                                'pivot' => $row['Pivot Point'] ?? null,
                                'r1' => $row['R1'] ?? null,
                                'r2' => $row['R2'] ?? null,
                                'r3' => $row['R3'] ?? null,
                                'uploaded' => $today,
                            ]
                        );
                        $inserted++;
                    }

                    flash("{$inserted} pivot calls uploaded, {$skipped} skipped (duplicates).", 'success');
                } catch (\Throwable $e) {
                    flash('Upload failed: ' . $e->getMessage(), 'danger');
                }

                redirect('/js-calls/upload-pivot');
            }

            $symbol = strtoupper(trim((string)request_input('symbol', '')));
            if ($symbol === '') {
                flash('Symbol is required.', 'warning');
                redirect('/js-calls/upload-pivot');
            }

            $exists = fetch_one(
                'SELECT id FROM js_pivot_calls WHERE symbol = :symbol AND DATE(uploaded_at) = :date',
                ['symbol' => $symbol, 'date' => $today]
            );
            if ($exists) {
                flash("Pivot call for {$symbol} already exists today.", 'warning');
                redirect('/js-calls/upload-pivot');
            }

            try {
                execute(
                    'INSERT INTO js_pivot_calls (symbol, close_price, volume, rsi_14, rsi_signal, macd_signal, s3, s2, s1, pivot, r1, r2, r3, uploaded_at)
                     VALUES (:symbol, :close, :volume, :rsi, :rsi_signal, :macd, :s3, :s2, :s1, :pivot, :r1, :r2, :r3, :uploaded)',
                    [
                        'symbol' => $symbol,
                        'close' => request_input('close_price'),
                        'volume' => request_input('volume'),
                        'rsi' => request_input('rsi_14'),
                        'rsi_signal' => request_input('rsi_signal'),
                        'macd' => request_input('macd_signal'),
                        's3' => request_input('s3'),
                        's2' => request_input('s2'),
                        's1' => request_input('s1'),
                        'pivot' => request_input('pivot'),
                        'r1' => request_input('r1'),
                        'r2' => request_input('r2'),
                        'r3' => request_input('r3'),
                        'uploaded' => $today,
                    ]
                );
                flash("Pivot call added for {$symbol}.", 'success');
            } catch (\Throwable $e) {
                flash('Manual entry failed: ' . $e->getMessage(), 'danger');
            }

            redirect('/js-calls/upload-pivot');
        }

        render('calls/upload_pivot_calls');
    }

    public static function listPivot(): void
    {
        $calls = fetch_all('SELECT * FROM js_pivot_calls ORDER BY uploaded_at DESC, symbol');
        render('calls/list_pivot_calls', ['calls' => $calls]);
    }

    public static function editPivot(string $id): void
    {
        $call = fetch_one('SELECT * FROM js_pivot_calls WHERE id = :id', ['id' => $id]);
        if (!$call) {
            redirect('/js-calls/pivot-calls');
        }

        if (request_method() === 'POST') {
            try {
                execute(
                    'UPDATE js_pivot_calls SET symbol = :symbol, close_price = :close, volume = :volume, rsi_14 = :rsi, rsi_signal = :rsi_signal, macd_signal = :macd, s3 = :s3, s2 = :s2, s1 = :s1, pivot = :pivot, r1 = :r1, r2 = :r2, r3 = :r3 WHERE id = :id',
                    [
                        'symbol' => strtoupper(trim((string)request_input('symbol', ''))),
                        'close' => request_input('close_price'),
                        'volume' => request_input('volume'),
                        'rsi' => request_input('rsi_14'),
                        'rsi_signal' => request_input('rsi_signal'),
                        'macd' => request_input('macd_signal'),
                        's3' => request_input('s3'),
                        's2' => request_input('s2'),
                        's1' => request_input('s1'),
                        'pivot' => request_input('pivot'),
                        'r1' => request_input('r1'),
                        'r2' => request_input('r2'),
                        'r3' => request_input('r3'),
                        'id' => $id,
                    ]
                );
                flash('Pivot call updated successfully.', 'success');
                redirect('/js-calls/pivot-calls');
            } catch (\Throwable $e) {
                flash('Update failed: ' . $e->getMessage(), 'danger');
            }
        }

        render('calls/edit_pivot_call', ['call' => $call]);
    }

    public static function deletePivot(string $id): void
    {
        execute('DELETE FROM js_pivot_calls WHERE id = :id', ['id' => $id]);
        flash('Pivot call deleted.', 'success');
        redirect('/js-calls/pivot-calls');
    }

    public static function dashboard(): void
    {
        $selectedDate = request_input('date');
        if ($selectedDate) {
            $date = date_create((string)$selectedDate);
            if (!$date) {
                flash('Invalid date format.', 'warning');
                redirect('/js-calls');
            }
            $dateFilter = $date->format('Y-m-d');
        } else {
            $dateFilter = date('Y-m-d');
        }

        $dailyCalls = fetch_all(
            'SELECT * FROM js_daily_calls WHERE DATE(uploaded_at) = :date ORDER BY symbol',
            ['date' => $dateFilter]
        );
        $pivotCalls = fetch_all(
            'SELECT * FROM js_pivot_calls WHERE DATE(uploaded_at) = :date ORDER BY symbol',
            ['date' => $dateFilter]
        );

        render('calls/js_calls_dashboard', [
            'date_filter' => $dateFilter,
            'daily_calls' => $dailyCalls,
            'pivot_calls' => $pivotCalls,
        ]);
    }

    private static function readCsv(string $path): array
    {
        $handle = fopen($path, 'r');
        if ($handle === false) {
            throw new \RuntimeException('Unable to open CSV file.');
        }

        $headers = fgetcsv($handle);
        if ($headers === false) {
            fclose($handle);
            return [];
        }

        $headers = array_map(static fn($h) => trim((string)$h), $headers);
        $rows = [];
        while (($row = fgetcsv($handle)) !== false) {
            if (count($row) === 1 && trim((string)$row[0]) === '') {
                continue;
            }
            $entry = [];
            foreach ($headers as $i => $header) {
                $entry[$header] = $row[$i] ?? null;
            }
            $rows[] = $entry;
        }
        fclose($handle);
        return $rows;
    }
}
