<?php

declare(strict_types=1);

namespace App\Controllers;

use DOMDocument;
use DOMXPath;

class MarketController
{
    public static function index(): void
    {
        redirect('/market/dashboard2');
    }

    public static function dashboard2(): void
    {
        $portfolioId = request_input('portfolio_id');
        $portfolioId = $portfolioId !== null ? (int)$portfolioId : null;

        $portfolios = fetch_all('SELECT * FROM portfolio ORDER BY id');

        $holdings = [];
        $symbols = [];
        if ($portfolioId) {
            $holdings = fetch_all(
                'SELECT h.*, s.symbol, s.sector, p.name AS portfolio_name
                 FROM holding h
                 JOIN stock s ON s.id = h.stock_id
                 JOIN portfolio p ON p.id = h.portfolio_id
                 WHERE h.portfolio_id = :pid',
                ['pid' => $portfolioId]
            );
            foreach ($holdings as $holding) {
                $symbols[] = strtoupper((string)$holding['symbol']);
            }
        }

        $marketData = self::fetchMarketData($symbols);

        $dashboardRows = [];
        $i = 1;
        foreach ($holdings as $holding) {
            $sym = strtoupper((string)$holding['symbol']);
            $sector = $holding['sector'] ?: 'N/A';
            $m = $marketData[$sym] ?? [];

            $priceNow = $m['current'] ?? (float)$holding['buy_price'];
            $amount = (int)$holding['quantity'] * (float)$holding['buy_price'];
            $valueNow = (int)$holding['quantity'] * (float)$priceNow;
            $pl = $valueNow - $amount;
            $plPct = $amount > 0 ? ($pl / $amount * 100) : 0.0;

            $txnRows = fetch_all(
                'SELECT date, type, price, quantity FROM `transaction` WHERE portfolio_id = :pid AND stock_id = :sid ORDER BY date DESC LIMIT 10',
                ['pid' => $portfolioId, 'sid' => $holding['stock_id']]
            );

            $buyTxns = [];
            $sellTxns = [];
            foreach ($txnRows as $txn) {
                $entry = [
                    'd' => date('Y-m-d', strtotime((string)$txn['date'])),
                    'p' => number_format((float)$txn['price'], 2, '.', ''),
                    'q' => (int)$txn['quantity'],
                ];
                $type = strtolower((string)$txn['type']);
                if ($type === 'buy' || $type === 'purchase') {
                    $buyTxns[] = $entry;
                } else {
                    $sellTxns[] = $entry;
                }
            }

            $daily = fetch_one('SELECT * FROM js_daily_calls WHERE symbol = :symbol ORDER BY uploaded_at DESC LIMIT 1', ['symbol' => $sym]);
            $pivot = fetch_one('SELECT * FROM js_pivot_calls WHERE symbol = :symbol ORDER BY uploaded_at DESC LIMIT 1', ['symbol' => $sym]);

            $dailyCard = $daily ? [
                'Close' => $daily['close_price'],
                'Pivot' => $daily['pivot'],
                'S1' => $daily['s1'],
                'R1' => $daily['r1'],
                'RSI' => $daily['rsi_14'],
                'Strat' => $daily['strategy'],
            ] : [];

            $pivotCard = $pivot ? [
                'Close' => $pivot['close_price'],
                'Vol' => $pivot['volume'],
                'RSI' => $pivot['rsi_14'],
                'RSI sig' => $pivot['rsi_signal'],
                'MACD' => $pivot['macd_signal'],
                'S1' => $pivot['s1'],
                'Pivot' => $pivot['pivot'],
                'R1' => $pivot['r1'],
            ] : [];

            $dashboardRows[] = [
                'sr' => $i++,
                'scrip' => $sym,
                'sector' => $sector,
                'qty' => (int)$holding['quantity'],
                'cost' => round((float)$holding['buy_price'], 2),
                'amount' => round($amount, 2),
                'last_close' => $m['ldcp'] ?? 'N/A',
                'yesterday' => $m['open'] ?? 'N/A',
                'today' => $priceNow,
                'change' => $m['change'] ?? 'N/A',
                'change_pct' => $m['change_pct'] ?? 'N/A',
                'today_pl' => round($pl, 2),
                'action' => '--',
                'current_value' => round($valueNow, 2),
                'pl' => round($pl, 2),
                'dividends' => 0.0,
                'total_pl' => round($pl, 2),
                'value' => round($valueNow, 2),
                'pl_pct' => number_format($plPct, 2) . '%',
                'pl_d_pct' => number_format($plPct, 2) . '%',
                'inv_pct' => '--',
                'portfolio' => $holding['portfolio_name'] ?? '',
                'dividend' => 0,
                'div_roi' => '--',
                'ideal_div' => '--',
                'buy_txns' => $buyTxns,
                'sell_txns' => $sellTxns,
                'daily_call' => $dailyCard,
                'pivot_call' => $pivotCard,
            ];
        }

        render('dashboard2', [
            'portfolios' => $portfolios,
            'selected_id' => $portfolioId,
            'dashboard_rows' => $dashboardRows,
        ]);
    }

    private static function fetchMarketData(array $symbols): array
    {
        if (empty($symbols)) {
            return [];
        }

        $url = 'https://dps.psx.com.pk/market-watch';
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_HTTPHEADER => ['User-Agent: Mozilla/5.0'],
        ]);
        $html = curl_exec($ch);
        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($status !== 200 || !is_string($html)) {
            return [];
        }

        $doc = new DOMDocument();
        @$doc->loadHTML($html);
        $xpath = new DOMXPath($doc);
        $table = $xpath->query("//table[contains(@class, 'tbl')]")->item(0);
        if (!$table) {
            return [];
        }

        $rows = $xpath->query('.//tbody/tr', $table);
        $marketData = [];
        $symbolSet = array_flip($symbols);

        foreach ($rows as $row) {
            $cells = $xpath->query('./td', $row);
            if ($cells->length < 11) {
                continue;
            }

            $symbolCell = $cells->item(0);
            $baseSymbol = trim((string)$symbolCell?->getAttribute('data-search'));
            if ($baseSymbol === '') {
                $baseSymbol = trim((string)$symbolCell?->getAttribute('data-order'));
            }
            if ($baseSymbol === '') {
                $strong = $xpath->query('.//strong', $symbolCell)->item(0);
                $baseSymbol = $strong ? trim((string)$strong->textContent) : '';
            }
            $baseSymbol = strtoupper($baseSymbol);
            if ($baseSymbol === '') {
                continue;
            }

            $matched = null;
            if (isset($symbolSet[$baseSymbol])) {
                $matched = $baseSymbol;
            } else {
                $first = strtoupper(strtok($baseSymbol, ' '));
                if (isset($symbolSet[$first])) {
                    $matched = $first;
                }
            }

            if (!$matched) {
                continue;
            }

            $tagDiv = $xpath->query(".//div[starts-with(@class,'tag--')]", $symbolCell)->item(0);
            $tag = $tagDiv ? strtoupper(trim((string)$tagDiv->textContent)) : '';

            $marketData[$matched] = [
                'ldcp' => self::safeFloat($cells->item(3)?->textContent),
                'open' => self::safeFloat($cells->item(4)?->textContent),
                'high' => self::safeFloat($cells->item(5)?->textContent),
                'low' => self::safeFloat($cells->item(6)?->textContent),
                'current' => self::safeFloat($cells->item(7)?->textContent),
                'change' => self::safeFloat($cells->item(8)?->textContent),
                'change_pct' => self::safeFloat(str_replace('%', '', (string)$cells->item(9)?->textContent)),
                'volume' => self::safeFloat($cells->item(10)?->textContent),
                'tag' => $tag,
            ];
        }

        return $marketData;
    }

    private static function safeFloat(?string $value): ?float
    {
        if ($value === null) {
            return null;
        }
        $cleaned = preg_replace('/[^0-9.\-]/', '', $value);
        if ($cleaned === '' || $cleaned === '-' || $cleaned === '.') {
            return null;
        }
        return (float)$cleaned;
    }
}
