<?php

declare(strict_types=1);

namespace App\Controllers;

class PortfolioController
{
    public static function index(): void
    {
        $portfolios = fetch_all('SELECT * FROM portfolio ORDER BY id');
        $stocks = fetch_all('SELECT * FROM stock ORDER BY symbol');

        $selectedId = request_input('portfolio_id');
        $selectedId = $selectedId !== null ? (int)$selectedId : null;
        if (!$selectedId && !empty($portfolios)) {
            $selectedId = (int)$portfolios[0]['id'];
        }

        $holdings = [];
        if ($selectedId) {
            $holdings = fetch_all(
                'SELECT h.*, s.symbol, s.sector FROM holding h JOIN stock s ON s.id = h.stock_id WHERE h.portfolio_id = :pid',
                ['pid' => $selectedId]
            );
        }

        render('portfolio', [
            'portfolios' => $portfolios,
            'stocks' => $stocks,
            'selected_id' => $selectedId,
            'holdings' => $holdings,
        ]);
    }

    public static function add(): void
    {
        $portfolioId = (int)request_input('portfolio_id');
        $symbol = strtoupper(trim((string)request_input('symbol', '')));
        $qty = (int)request_input('qty');
        $price = (float)request_input('price');

        $stock = fetch_one('SELECT id FROM stock WHERE symbol = :symbol', ['symbol' => $symbol]);
        if (!$stock) {
            flash("Stock symbol '{$symbol}' not found in database.", 'danger');
            redirect('/portfolio?portfolio_id=' . $portfolioId);
        }

        execute(
            'INSERT INTO holding (portfolio_id, stock_id, quantity, buy_price) VALUES (:pid, :sid, :qty, :price)',
            [
                'pid' => $portfolioId,
                'sid' => $stock['id'],
                'qty' => $qty,
                'price' => $price,
            ]
        );

        redirect('/portfolio?portfolio_id=' . $portfolioId);
    }

    public static function delete(string $id): void
    {
        $holding = fetch_one('SELECT * FROM holding WHERE id = :id', ['id' => $id]);
        if (!$holding) {
            redirect('/portfolio');
        }

        execute('DELETE FROM holding WHERE id = :id', ['id' => $id]);
        redirect('/portfolio?portfolio_id=' . $holding['portfolio_id']);
    }

    public static function edit(string $id): void
    {
        $holding = fetch_one(
            'SELECT h.*, s.symbol, s.sector FROM holding h JOIN stock s ON s.id = h.stock_id WHERE h.id = :id',
            ['id' => $id]
        );
        if (!$holding) {
            redirect('/portfolio');
        }

        $stocks = fetch_all('SELECT * FROM stock ORDER BY symbol');

        if (request_method() === 'POST') {
            $symbol = strtoupper(trim((string)request_input('symbol', '')));
            $stock = fetch_one('SELECT id FROM stock WHERE symbol = :symbol', ['symbol' => $symbol]);
            if (!$stock) {
                flash("Stock symbol '{$symbol}' not found.", 'danger');
                redirect('/portfolio?portfolio_id=' . $holding['portfolio_id']);
            }

            $qty = (int)request_input('qty');
            $price = (float)request_input('price');

            execute(
                'UPDATE holding SET stock_id = :sid, quantity = :qty, buy_price = :price WHERE id = :id',
                [
                    'sid' => $stock['id'],
                    'qty' => $qty,
                    'price' => $price,
                    'id' => $id,
                ]
            );

            redirect('/portfolio?portfolio_id=' . $holding['portfolio_id']);
        }

        render('edit_holding', [
            'holding' => $holding,
            'stocks' => $stocks,
        ]);
    }

    public static function import(): void
    {
        $portfolioId = (int)request_input('portfolio_id');
        $file = $_FILES['file'] ?? null;

        if (!$file || $file['error'] !== UPLOAD_ERR_OK) {
            flash('No file selected.', 'danger');
            redirect('/portfolio?portfolio_id=' . $portfolioId);
        }

        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        execute('DELETE FROM holding WHERE portfolio_id = :pid', ['pid' => $portfolioId]);

        try {
            if ($ext === 'csv') {
                $handle = fopen($file['tmp_name'], 'r');
                if (!$handle) {
                    throw new \RuntimeException('Could not read CSV file.');
                }
                $headers = fgetcsv($handle);
                if (!$headers) {
                    throw new \RuntimeException('CSV is empty.');
                }
                while (($row = fgetcsv($handle)) !== false) {
                    $data = array_combine($headers, $row);
                    self::createHoldingRow($portfolioId, $data);
                }
                fclose($handle);
            } elseif ($ext === 'json') {
                $raw = file_get_contents($file['tmp_name']);
                $data = json_decode($raw, true);
                if (!is_array($data)) {
                    throw new \RuntimeException('Invalid JSON format. Must be a list of holdings.');
                }
                foreach ($data as $row) {
                    self::createHoldingRow($portfolioId, $row);
                }
            } else {
                flash('Unsupported file format. Only CSV and JSON are accepted.', 'warning');
                redirect('/portfolio?portfolio_id=' . $portfolioId);
            }

            flash('Portfolio holdings imported successfully.', 'success');
        } catch (\Throwable $e) {
            flash('Import failed: ' . $e->getMessage(), 'danger');
        }

        redirect('/portfolio?portfolio_id=' . $portfolioId);
    }

    private static function createHoldingRow(int $portfolioId, array $row): void
    {
        $symbol = strtoupper(trim((string)($row['symbol'] ?? '')));
        $stock = fetch_one('SELECT id FROM stock WHERE symbol = :symbol', ['symbol' => $symbol]);
        if (!$stock) {
            throw new \RuntimeException("Symbol '{$symbol}' not found in Stock table.");
        }

        $qty = (int)($row['qty'] ?? $row['quantity'] ?? 0);
        $price = (float)($row['buy_price'] ?? 0);

        execute(
            'INSERT INTO holding (portfolio_id, stock_id, quantity, buy_price) VALUES (:pid, :sid, :qty, :price)',
            [
                'pid' => $portfolioId,
                'sid' => $stock['id'],
                'qty' => $qty,
                'price' => $price,
            ]
        );
    }
}
