<?php

declare(strict_types=1);

namespace App\Controllers;

class StocksController
{
    public static function index(): void
    {
        $stocks = fetch_all('SELECT * FROM stock ORDER BY symbol');
        render('stocks/index', ['stocks' => $stocks]);
    }

    public static function add(): void
    {
        $symbol = strtoupper(trim((string)request_input('symbol', '')));
        $sector = trim((string)request_input('sector', ''));

        if ($symbol === '') {
            flash('Symbol is required.', 'warning');
            redirect('/stocks');
        }

        $existing = fetch_one('SELECT id FROM stock WHERE symbol = :symbol', ['symbol' => $symbol]);
        if ($existing) {
            flash('Stock already exists.', 'warning');
            redirect('/stocks');
        }

        execute('INSERT INTO stock (symbol, sector) VALUES (:symbol, :sector)', [
            'symbol' => $symbol,
            'sector' => $sector,
        ]);
        flash('Stock added successfully.', 'success');
        redirect('/stocks');
    }

    public static function uploadJson(): void
    {
        $file = $_FILES['file'] ?? null;
        if (!$file || $file['error'] !== UPLOAD_ERR_OK || !str_ends_with($file['name'], '.json')) {
            flash('Invalid file format. Please upload a JSON file.', 'danger');
            redirect('/stocks');
        }

        try {
            $data = json_decode((string)file_get_contents($file['tmp_name']), true, 512, JSON_THROW_ON_ERROR);
            $added = 0;
            foreach ($data as $entry) {
                $symbol = strtoupper(trim((string)($entry['symbol'] ?? '')));
                $sector = trim((string)($entry['sector'] ?? ''));
                if ($symbol === '') {
                    continue;
                }
                $exists = fetch_one('SELECT id FROM stock WHERE symbol = :symbol', ['symbol' => $symbol]);
                if (!$exists) {
                    execute('INSERT INTO stock (symbol, sector) VALUES (:symbol, :sector)', [
                        'symbol' => $symbol,
                        'sector' => $sector,
                    ]);
                    $added++;
                }
            }
            flash($added . ' stocks imported successfully.', 'success');
        } catch (\Throwable $e) {
            flash('Error reading JSON: ' . $e->getMessage(), 'danger');
        }

        redirect('/stocks');
    }

    public static function delete(string $id): void
    {
        execute('DELETE FROM stock WHERE id = :id', ['id' => $id]);
        flash('Stock deleted.', 'success');
        redirect('/stocks');
    }
}
