<?php

declare(strict_types=1);

namespace App\Controllers;

use DateTime;

class TransactionsController
{
    public static function index(): void
    {
        $portfolios = fetch_all('SELECT * FROM portfolio ORDER BY id');
        $stocks = fetch_all('SELECT * FROM stock ORDER BY symbol');

        $selectedId = request_input('portfolio_id');
        $selectedId = $selectedId !== null ? (int)$selectedId : null;

        $transactions = [];
        $holdings = [];
        $dividends = [];

        if ($selectedId) {
            $transactions = fetch_all(
                'SELECT t.*, s.symbol, s.sector FROM `transaction` t JOIN stock s ON s.id = t.stock_id WHERE t.portfolio_id = :pid ORDER BY t.date DESC',
                ['pid' => $selectedId]
            );
            $holdings = fetch_all(
                'SELECT h.*, s.symbol, s.sector FROM holding h JOIN stock s ON s.id = h.stock_id WHERE h.portfolio_id = :pid',
                ['pid' => $selectedId]
            );
            $dividends = fetch_all(
                'SELECT * FROM dividend WHERE portfolio_id = :pid ORDER BY payment_date DESC',
                ['pid' => $selectedId]
            );
        }

        render('transaction_form', [
            'portfolios' => $portfolios,
            'stocks' => $stocks,
            'transactions' => $transactions,
            'selected_portfolio_id' => $selectedId,
            'holdings' => $holdings,
            'dividends' => $dividends,
        ]);
    }

    public static function add(): void
    {
        $portfolioId = (int)request_input('portfolio_id');
        $symbol = strtoupper(trim((string)request_input('symbol', '')));
        $type = (string)request_input('type');
        $quantity = (int)request_input('qty');
        $price = (float)request_input('price');

        $stock = fetch_one('SELECT id FROM stock WHERE symbol = :symbol', ['symbol' => $symbol]);
        if (!$stock) {
            flash("Stock symbol '{$symbol}' not found.", 'danger');
            redirect('/transaction?portfolio_id=' . $portfolioId);
        }

        db()->beginTransaction();
        try {
            execute(
                'INSERT INTO `transaction` (portfolio_id, stock_id, type, quantity, price, date) VALUES (:pid, :sid, :type, :qty, :price, :date)',
                [
                    'pid' => $portfolioId,
                    'sid' => $stock['id'],
                    'type' => $type,
                    'qty' => $quantity,
                    'price' => $price,
                    'date' => date('Y-m-d H:i:s'),
                ]
            );

            $holding = fetch_one(
                'SELECT * FROM holding WHERE portfolio_id = :pid AND stock_id = :sid',
                ['pid' => $portfolioId, 'sid' => $stock['id']]
            );

            if ($type === 'buy') {
                if ($holding) {
                    $totalQty = (int)$holding['quantity'] + $quantity;
                    $avgPrice = (((float)$holding['quantity'] * (float)$holding['buy_price']) + ($quantity * $price)) / $totalQty;
                    execute(
                        'UPDATE holding SET quantity = :qty, buy_price = :price WHERE id = :id',
                        ['qty' => $totalQty, 'price' => $avgPrice, 'id' => $holding['id']]
                    );
                } else {
                    execute(
                        'INSERT INTO holding (portfolio_id, stock_id, quantity, buy_price) VALUES (:pid, :sid, :qty, :price)',
                        ['pid' => $portfolioId, 'sid' => $stock['id'], 'qty' => $quantity, 'price' => $price]
                    );
                }
            } elseif ($type === 'sell' && $holding) {
                $newQty = (int)$holding['quantity'] - $quantity;
                if ($newQty <= 0) {
                    execute('DELETE FROM holding WHERE id = :id', ['id' => $holding['id']]);
                } else {
                    execute('UPDATE holding SET quantity = :qty WHERE id = :id', ['qty' => $newQty, 'id' => $holding['id']]);
                }
            }

            db()->commit();
        } catch (\Throwable $e) {
            db()->rollBack();
            flash('Failed to record transaction: ' . $e->getMessage(), 'danger');
        }

        redirect('/transaction?portfolio_id=' . $portfolioId);
    }

    public static function delete(string $id): void
    {
        $tx = fetch_one('SELECT * FROM `transaction` WHERE id = :id', ['id' => $id]);
        if (!$tx) {
            redirect('/transaction');
        }

        $holding = fetch_one(
            'SELECT * FROM holding WHERE portfolio_id = :pid AND stock_id = :sid',
            ['pid' => $tx['portfolio_id'], 'sid' => $tx['stock_id']]
        );

        db()->beginTransaction();
        try {
            if ($holding) {
                if ($tx['type'] === 'buy') {
                    $newQty = (int)$holding['quantity'] - (int)$tx['quantity'];
                    if ($newQty <= 0) {
                        execute('DELETE FROM holding WHERE id = :id', ['id' => $holding['id']]);
                    } else {
                        execute('UPDATE holding SET quantity = :qty WHERE id = :id', ['qty' => $newQty, 'id' => $holding['id']]);
                    }
                } elseif ($tx['type'] === 'sell') {
                    $newQty = (int)$holding['quantity'] + (int)$tx['quantity'];
                    execute('UPDATE holding SET quantity = :qty WHERE id = :id', ['qty' => $newQty, 'id' => $holding['id']]);
                }
            }

            execute('DELETE FROM `transaction` WHERE id = :id', ['id' => $id]);
            db()->commit();
        } catch (\Throwable $e) {
            db()->rollBack();
            flash('Delete failed: ' . $e->getMessage(), 'danger');
        }

        redirect('/transaction?portfolio_id=' . $tx['portfolio_id']);
    }

    public static function edit(string $id): void
    {
        $tx = fetch_one('SELECT * FROM `transaction` WHERE id = :id', ['id' => $id]);
        if (!$tx) {
            redirect('/transaction');
        }

        $portfolios = fetch_all('SELECT * FROM portfolio ORDER BY id');
        $stocks = fetch_all('SELECT * FROM stock ORDER BY symbol');

        if (request_method() === 'POST') {
            $newPortfolioId = (int)request_input('portfolio_id');
            $newSymbol = strtoupper(trim((string)request_input('symbol', '')));
            $newType = (string)request_input('type');
            $newQty = (int)request_input('qty');
            $newPrice = (float)request_input('price');

            $stock = fetch_one('SELECT id FROM stock WHERE symbol = :symbol', ['symbol' => $newSymbol]);
            if (!$stock) {
                flash("Stock symbol '{$newSymbol}' not found.", 'danger');
                redirect('/transaction?portfolio_id=' . $newPortfolioId);
            }

            db()->beginTransaction();
            try {
                $oldHolding = fetch_one(
                    'SELECT * FROM holding WHERE portfolio_id = :pid AND stock_id = :sid',
                    ['pid' => $tx['portfolio_id'], 'sid' => $tx['stock_id']]
                );

                if ($oldHolding) {
                    if ($tx['type'] === 'buy') {
                        $newQtyOld = (int)$oldHolding['quantity'] - (int)$tx['quantity'];
                        if ($newQtyOld <= 0) {
                            execute('DELETE FROM holding WHERE id = :id', ['id' => $oldHolding['id']]);
                        } else {
                            execute('UPDATE holding SET quantity = :qty WHERE id = :id', ['qty' => $newQtyOld, 'id' => $oldHolding['id']]);
                        }
                    } elseif ($tx['type'] === 'sell') {
                        $newQtyOld = (int)$oldHolding['quantity'] + (int)$tx['quantity'];
                        execute('UPDATE holding SET quantity = :qty WHERE id = :id', ['qty' => $newQtyOld, 'id' => $oldHolding['id']]);
                    }
                }

                $newHolding = fetch_one(
                    'SELECT * FROM holding WHERE portfolio_id = :pid AND stock_id = :sid',
                    ['pid' => $newPortfolioId, 'sid' => $stock['id']]
                );

                if ($newType === 'buy') {
                    if ($newHolding) {
                        $totalQty = (int)$newHolding['quantity'] + $newQty;
                        $avgPrice = (((float)$newHolding['quantity'] * (float)$newHolding['buy_price']) + ($newQty * $newPrice)) / $totalQty;
                        execute(
                            'UPDATE holding SET quantity = :qty, buy_price = :price WHERE id = :id',
                            ['qty' => $totalQty, 'price' => $avgPrice, 'id' => $newHolding['id']]
                        );
                    } else {
                        execute(
                            'INSERT INTO holding (portfolio_id, stock_id, quantity, buy_price) VALUES (:pid, :sid, :qty, :price)',
                            ['pid' => $newPortfolioId, 'sid' => $stock['id'], 'qty' => $newQty, 'price' => $newPrice]
                        );
                    }
                } elseif ($newType === 'sell' && $newHolding) {
                    $updatedQty = (int)$newHolding['quantity'] - $newQty;
                    if ($updatedQty <= 0) {
                        execute('DELETE FROM holding WHERE id = :id', ['id' => $newHolding['id']]);
                    } else {
                        execute('UPDATE holding SET quantity = :qty WHERE id = :id', ['qty' => $updatedQty, 'id' => $newHolding['id']]);
                    }
                }

                execute(
                    'UPDATE `transaction` SET portfolio_id = :pid, stock_id = :sid, type = :type, quantity = :qty, price = :price WHERE id = :id',
                    [
                        'pid' => $newPortfolioId,
                        'sid' => $stock['id'],
                        'type' => $newType,
                        'qty' => $newQty,
                        'price' => $newPrice,
                        'id' => $id,
                    ]
                );

                db()->commit();
            } catch (\Throwable $e) {
                db()->rollBack();
                flash('Update failed: ' . $e->getMessage(), 'danger');
            }

            redirect('/transaction?portfolio_id=' . $newPortfolioId);
        }

        render('edit_transaction', [
            'tx' => $tx,
            'portfolios' => $portfolios,
            'stocks' => $stocks,
        ]);
    }

    public static function uploadSalesExcel(): void
    {
        $file = $_FILES['file'] ?? null;
        $portfolioId = (int)request_input('portfolio_id');

        if (!$file || $file['error'] !== UPLOAD_ERR_OK || !str_ends_with(strtolower($file['name']), '.csv')) {
            flash('Invalid file format. Please upload a .csv file.', 'error');
            redirect('/transaction?portfolio_id=' . $portfolioId);
        }

        try {
            $rows = self::readCsv($file['tmp_name']);
            db()->beginTransaction();

            foreach ($rows as $row) {
                $symbol = strtoupper(trim((string)($row['Script'] ?? '')));
                if ($symbol === '') {
                    continue;
                }
                $quantity = (int)($row['Quantity'] ?? 0);
                $rate = (float)($row['Rate'] ?? 0);
                $commPerUnit = (float)($row['Comm.'] ?? 0);
                $sst = (float)($row['SST'] ?? 0);
                $settleRaw = $row['Settle. Date'] ?? null;
                $settleDate = self::parseDate($settleRaw);

                $stock = fetch_one('SELECT id FROM stock WHERE symbol = :symbol', ['symbol' => $symbol]);
                if (!$stock) {
                    flash("Stock '{$symbol}' not found.", 'danger');
                    continue;
                }

                $gross = $quantity * $rate;
                $totalComm = $quantity * $commPerUnit;

                $holding = fetch_one(
                    'SELECT * FROM holding WHERE portfolio_id = :pid AND stock_id = :sid',
                    ['pid' => $portfolioId, 'sid' => $stock['id']]
                );
                $buyPrice = $holding ? (float)$holding['buy_price'] : 0.0;
                $costBasis = $quantity * $buyPrice;
                $profit = $gross - $costBasis;
                $cgt = $profit > 0 ? $profit * 0.15 : 0.0;
                $netSale = $gross - $totalComm - $sst - $cgt;
                $unitNetPrice = $quantity ? round($netSale / $quantity, 2) : 0.0;

                execute(
                    'INSERT INTO `transaction` (portfolio_id, stock_id, type, quantity, price, gross_amount, net_amount, commission, sst, cgt, date)
                     VALUES (:pid, :sid, :type, :qty, :price, :gross, :net, :comm, :sst, :cgt, :date)',
                    [
                        'pid' => $portfolioId,
                        'sid' => $stock['id'],
                        'type' => 'sell',
                        'qty' => $quantity,
                        'price' => $unitNetPrice,
                        'gross' => $gross,
                        'net' => $netSale,
                        'comm' => round($totalComm, 2),
                        'sst' => round($sst, 2),
                        'cgt' => round($cgt, 2),
                        'date' => $settleDate,
                    ]
                );

                if ($holding) {
                    $newQty = (int)$holding['quantity'] - $quantity;
                    if ($newQty <= 0) {
                        execute('DELETE FROM holding WHERE id = :id', ['id' => $holding['id']]);
                    } else {
                        execute('UPDATE holding SET quantity = :qty WHERE id = :id', ['qty' => $newQty, 'id' => $holding['id']]);
                    }
                }
            }

            db()->commit();
            flash('CSV uploaded and sale transactions recorded successfully!', 'success');
        } catch (\Throwable $e) {
            db()->rollBack();
            flash('Error processing CSV file: ' . $e->getMessage(), 'danger');
        }

        redirect('/transaction?portfolio_id=' . $portfolioId);
    }

    public static function uploadPurchaseExcel(): void
    {
        $file = $_FILES['file'] ?? null;
        $portfolioId = (int)request_input('portfolio_id');

        if (!$file || $file['error'] !== UPLOAD_ERR_OK || !str_ends_with(strtolower($file['name']), '.csv')) {
            flash('Invalid file format. Please upload a .csv file.', 'error');
            redirect('/transaction?portfolio_id=' . $portfolioId);
        }

        try {
            $rows = self::readCsv($file['tmp_name']);
            db()->beginTransaction();

            foreach ($rows as $row) {
                $symbol = strtoupper(trim((string)($row['Script'] ?? '')));
                if ($symbol === '') {
                    continue;
                }
                $quantity = (int)($row['Quantity'] ?? 0);
                $rate = (float)($row['Rate'] ?? 0);
                $settleRaw = $row['Settle. Date'] ?? null;
                $settleDate = self::parseDate($settleRaw);
                $commPerUnit = (float)($row['Comm.'] ?? 0);
                $sst = (float)($row['SST'] ?? 0);

                $stock = fetch_one('SELECT id FROM stock WHERE symbol = :symbol', ['symbol' => $symbol]);
                if (!$stock) {
                    execute('INSERT INTO stock (symbol, sector) VALUES (:symbol, NULL)', ['symbol' => $symbol]);
                    $stock = ['id' => (int)db()->lastInsertId()];
                }

                $gross = $quantity * $rate;
                $totalComm = $quantity * $commPerUnit;
                $netCost = $gross + $totalComm + $sst;
                $unitNetPrice = $quantity ? round($netCost / $quantity, 2) : 0.0;

                execute(
                    'INSERT INTO `transaction` (portfolio_id, stock_id, type, quantity, price, gross_amount, net_amount, commission, sst, cgt, date)
                     VALUES (:pid, :sid, :type, :qty, :price, :gross, :net, :comm, :sst, :cgt, :date)',
                    [
                        'pid' => $portfolioId,
                        'sid' => $stock['id'],
                        'type' => 'buy',
                        'qty' => $quantity,
                        'price' => $unitNetPrice,
                        'gross' => $gross,
                        'net' => $netCost,
                        'comm' => round($totalComm, 2),
                        'sst' => round($sst, 2),
                        'cgt' => 0.0,
                        'date' => $settleDate,
                    ]
                );

                $holding = fetch_one(
                    'SELECT * FROM holding WHERE portfolio_id = :pid AND stock_id = :sid',
                    ['pid' => $portfolioId, 'sid' => $stock['id']]
                );

                if ($holding) {
                    $totalQty = (int)$holding['quantity'] + $quantity;
                    $avgPrice = (((float)$holding['quantity'] * (float)$holding['buy_price']) + ($quantity * $unitNetPrice)) / $totalQty;
                    execute(
                        'UPDATE holding SET quantity = :qty, buy_price = :price WHERE id = :id',
                        ['qty' => $totalQty, 'price' => $avgPrice, 'id' => $holding['id']]
                    );
                } else {
                    execute(
                        'INSERT INTO holding (portfolio_id, stock_id, quantity, buy_price) VALUES (:pid, :sid, :qty, :price)',
                        ['pid' => $portfolioId, 'sid' => $stock['id'], 'qty' => $quantity, 'price' => $unitNetPrice]
                    );
                }
            }

            db()->commit();
            flash('Purchase CSV uploaded and new stocks added successfully.', 'success');
        } catch (\Throwable $e) {
            db()->rollBack();
            flash('Upload failed: ' . $e->getMessage(), 'danger');
        }

        redirect('/transaction?portfolio_id=' . $portfolioId);
    }

    public static function uploadTransactionExcel(): void
    {
        $file = $_FILES['file'] ?? null;
        $portfolioId = (int)request_input('portfolio_id');

        if (!$file || $file['error'] !== UPLOAD_ERR_OK || !str_ends_with(strtolower($file['name']), '.csv')) {
            flash('Invalid file format. Please upload a .csv file.', 'error');
            redirect('/transaction?portfolio_id=' . $portfolioId);
        }

        try {
            $rows = self::readCsv($file['tmp_name']);
        } catch (\Throwable $e) {
            flash('Could not read CSV file: ' . $e->getMessage(), 'danger');
            redirect('/transaction?portfolio_id=' . $portfolioId);
        }

        $tradeMap = [
            'b' => 'buy', 'buy' => 'buy', 'purchase' => 'buy', 'p' => 'buy',
            's' => 'sell', 'sell' => 'sell', 'sale' => 'sell', 'sold' => 'sell',
        ];

        foreach ($rows as &$row) {
            $raw = strtolower(preg_replace('/\s+/', '', (string)($row['TradeType'] ?? '')));
            $row['TradeTypeNorm'] = $tradeMap[$raw] ?? null;
        }
        unset($row);

        $bad = array_filter($rows, static fn($r) => empty($r['TradeTypeNorm']));
        if (!empty($bad)) {
            $vals = array_unique(array_map(static fn($r) => (string)($r['TradeType'] ?? ''), $bad));
            flash('Unrecognized TradeType values found: ' . implode(', ', $vals), 'danger');
            redirect('/transaction?portfolio_id=' . $portfolioId);
        }

        db()->beginTransaction();
        try {
            foreach ($rows as $row) {
                $tradeType = (string)$row['TradeTypeNorm'];
                $symbol = strtoupper(trim((string)($row['SecuritySymbol'] ?? '')));
                if ($symbol === '') {
                    continue;
                }
                $quantity = (int)($row['Quantity'] ?? 0);
                $rate = (float)($row['Rate'] ?? 0);
                $grossAmt = (float)($row['GrossAmount'] ?? 0);
                $netAmt = (float)($row['NetAmount'] ?? 0);

                $stock = fetch_one('SELECT id FROM stock WHERE symbol = :symbol', ['symbol' => $symbol]);
                if (!$stock) {
                    execute('INSERT INTO stock (symbol, sector) VALUES (:symbol, NULL)', ['symbol' => $symbol]);
                    $stock = ['id' => (int)db()->lastInsertId()];
                }

                $holding = fetch_one(
                    'SELECT * FROM holding WHERE portfolio_id = :pid AND stock_id = :sid',
                    ['pid' => $portfolioId, 'sid' => $stock['id']]
                );

                $commission = (float)($row['Brokerage'] ?? 0);
                $sst = (float)($row['SalesTaxonBrokerage'] ?? 0) +
                       (float)($row['NCCPLIDSFee'] ?? 0) +
                       (float)($row['NCCPLRMSFee'] ?? 0);
                $cgt = (float)($row['CGT'] ?? 0);

                $realizedPl = 0.0;
                $costPerShare = 0.0;

                if ($tradeType === 'buy') {
                    $costPerShare = $quantity ? ($netAmt / $quantity) : 0.0;
                } elseif ($tradeType === 'sell' && $holding) {
                    $sellPricePerShare = $quantity ? ($netAmt / $quantity) : 0.0;
                    $realizedPl = ($sellPricePerShare - (float)$holding['buy_price']) * $quantity;
                }

                execute(
                    'INSERT INTO `transaction` (portfolio_id, stock_id, type, quantity, price, gross_amount, net_amount, commission, sst, cgt, date, realized_pl)
                     VALUES (:pid, :sid, :type, :qty, :price, :gross, :net, :comm, :sst, :cgt, :date, :realized)',
                    [
                        'pid' => $portfolioId,
                        'sid' => $stock['id'],
                        'type' => $tradeType,
                        'qty' => $quantity,
                        'price' => $rate,
                        'gross' => $grossAmt,
                        'net' => $netAmt,
                        'comm' => $commission,
                        'sst' => round($sst, 2),
                        'cgt' => $cgt,
                        'date' => self::parseDate($row['TradeDate'] ?? null),
                        'realized' => $realizedPl,
                    ]
                );

                if (!$holding) {
                    execute(
                        'INSERT INTO holding (portfolio_id, stock_id, quantity, buy_price) VALUES (:pid, :sid, :qty, :price)',
                        ['pid' => $portfolioId, 'sid' => $stock['id'], 'qty' => 0, 'price' => 0]
                    );
                    $holding = ['quantity' => 0, 'buy_price' => 0, 'id' => (int)db()->lastInsertId()];
                }

                if ($tradeType === 'buy') {
                    if ((int)$holding['quantity'] < 0) {
                        $net = (int)$holding['quantity'] + $quantity;
                        if ($net < 0) {
                            execute('UPDATE holding SET quantity = :qty WHERE id = :id', ['qty' => $net, 'id' => $holding['id']]);
                        } elseif ($net === 0) {
                            execute('UPDATE holding SET quantity = 0, buy_price = 0 WHERE id = :id', ['id' => $holding['id']]);
                        } else {
                            execute('UPDATE holding SET quantity = :qty, buy_price = :price WHERE id = :id', ['qty' => $net, 'price' => $costPerShare, 'id' => $holding['id']]);
                        }
                    } else {
                        $totalCost = ((float)$holding['quantity'] * (float)$holding['buy_price']) + ($quantity * $costPerShare);
                        $newQty = (int)$holding['quantity'] + $quantity;
                        $newPrice = $newQty ? round($totalCost / $newQty, 6) : 0.0;
                        execute(
                            'UPDATE holding SET quantity = :qty, buy_price = :price WHERE id = :id',
                            ['qty' => $newQty, 'price' => $newPrice, 'id' => $holding['id']]
                        );
                    }
                } elseif ($tradeType === 'sell') {
                    $newQty = (int)$holding['quantity'] - $quantity;
                    $params = ['qty' => $newQty, 'id' => $holding['id']];
                    if ($newQty === 0) {
                        execute('UPDATE holding SET quantity = :qty, buy_price = 0 WHERE id = :id', $params);
                    } else {
                        execute('UPDATE holding SET quantity = :qty WHERE id = :id', $params);
                    }
                }
            }

            db()->commit();
            flash('Transactions uploaded successfully!', 'success');
        } catch (\Throwable $e) {
            db()->rollBack();
            flash('Upload failed: ' . $e->getMessage(), 'danger');
        }

        redirect('/transaction?portfolio_id=' . $portfolioId);
    }

    private static function readCsv(string $path): array
    {
        $handle = fopen($path, 'r');
        if ($handle === false) {
            throw new \RuntimeException('Unable to open CSV file.');
        }

        $headers = fgetcsv($handle);
        if ($headers === false) {
            fclose($handle);
            return [];
        }

        $headers = array_map(static fn($h) => trim((string)$h), $headers);
        $rows = [];
        while (($row = fgetcsv($handle)) !== false) {
            if (count($row) === 1 && trim((string)$row[0]) === '') {
                continue;
            }
            $entry = [];
            foreach ($headers as $i => $header) {
                $entry[$header] = $row[$i] ?? null;
            }
            $rows[] = $entry;
        }
        fclose($handle);
        return $rows;
    }

    private static function parseDate(mixed $value): string
    {
        if ($value instanceof DateTime) {
            return $value->format('Y-m-d');
        }
        if (is_numeric($value)) {
            $date = DateTime::createFromFormat('Y-m-d', '1899-12-30');
            if ($date) {
                $date->modify('+' . (int)$value . ' days');
                return $date->format('Y-m-d');
            }
        }
        $date = date_create((string)$value);
        return $date ? $date->format('Y-m-d') : date('Y-m-d');
    }
}
